package com.trafficparrot.examples.hello.ibmmq;

import com.ibm.mq.*;
import com.ibm.mq.constants.CMQC;

import java.io.IOException;
import java.util.Hashtable;

import static com.ibm.mq.constants.CMQC.*;
import static org.apache.commons.lang3.StringUtils.left;

public class IbmMqClient {
    private static MQQueueManager queueManager() throws MQException {
        Hashtable<Object, Object> connectionProperties = new Hashtable<>();
        connectionProperties.put(CMQC.HOST_NAME_PROPERTY, System.getProperty("ibm.mq.host", "localhost"));
        connectionProperties.put(CMQC.PORT_PROPERTY, Integer.parseInt(System.getProperty("ibm.mq.port", "1414")));
        connectionProperties.put(CMQC.CHANNEL_PROPERTY, System.getProperty("ibm.mq.channel", "DEV.APP.SVRCONN"));
        connectionProperties.put(CMQC.USER_ID_PROPERTY, System.getProperty("ibm.mq.user", "app"));
        boolean useMqcsp = Boolean.parseBoolean(System.getProperty("ibm.mq.use.mqcsp", "false"));
        connectionProperties.put(CMQC.USE_MQCSP_AUTHENTICATION_PROPERTY, useMqcsp);
        if (useMqcsp) {
            connectionProperties.put(CMQC.PASSWORD_PROPERTY, System.getProperty("ibm.mq.password", ""));
        } else {
            connectionProperties.put(CMQC.PASSWORD_PROPERTY, left(System.getProperty("ibm.mq.password", ""), 12));
        }
        return new MQQueueManager(System.getProperty("ibm.mq.qm", "QM1"), connectionProperties);
    }

    public void sendIbmMqMessageToQueue(String queueName, String text) throws MQException, IOException {
        sendIbmMqMessageTo(true, queueName, text);
    }

    public void sendIbmMqMessageToTopic(String topicName, String text) throws MQException, IOException {
        sendIbmMqMessageTo(false, topicName, text);
    }

    private void sendIbmMqMessageTo(boolean queue, String name, String text) throws MQException, IOException {
        MQQueueManager mqQueueManager = queueManager();
        try {
            MQDestination mqDestination = mqDestination(queue, name, mqQueueManager);
            try {
                MQMessage mqMessage = new MQMessage();
                mqMessage.format = CMQC.MQFMT_STRING;
                mqMessage.writeString(text);
                MQPutMessageOptions mqPutMessageOptions = new MQPutMessageOptions();
                mqPutMessageOptions.options = CMQC.MQPMO_NEW_CORREL_ID | CMQC.MQPMO_NEW_MSG_ID;
                mqDestination.put(mqMessage, mqPutMessageOptions);
            } finally {
                mqDestination.close();
            }
        } finally {
            mqQueueManager.disconnect();
            mqQueueManager.close();
        }
    }

    private MQDestination mqDestination(boolean queue, String destinationName, MQQueueManager mqQueueManager) throws MQException {
        if (queue) {
            return mqQueueManager.accessQueue(destinationName, MQOO_OUTPUT);
        } else {
            return mqQueueManager.accessTopic(destinationName, "", MQTOPIC_OPEN_AS_PUBLICATION, MQOO_OUTPUT);
        }
    }

    public String receiveIbmMqMessageFromQueue(String queueName) throws MQException, IOException {
        MQQueueManager mqQueueManager = queueManager();
        try {
            MQQueue mqQueue = mqQueueManager.accessQueue(queueName, MQOO_INPUT_SHARED);
            try {
                MQMessage mqMessage = new MQMessage();
                MQGetMessageOptions mqGetMessageOptions = new MQGetMessageOptions();
                mqGetMessageOptions.options = CMQC.MQGMO_WAIT | CMQC.MQGMO_FAIL_IF_QUIESCING;
                mqGetMessageOptions.waitInterval = 10_000;
                mqQueue.get(mqMessage, mqGetMessageOptions);
                return mqMessage.readStringOfByteLength(mqMessage.getDataLength());
            } finally {
                mqQueue.close();
            }
        } finally {
            mqQueueManager.disconnect();
            mqQueueManager.close();
        }
    }

    public WaitForOneIbmMqMessage listenOnIbmMqTopic(String topicName) throws MQException {
        MQQueueManager mqQueueManager = queueManager();
        MQTopic mqTopic = mqQueueManager.accessTopic(topicName, "", CMQC.MQTOPIC_OPEN_AS_SUBSCRIPTION, CMQC.MQSO_CREATE);
        return new WaitForOneIbmMqMessage(mqQueueManager, mqTopic);
    }
}
