package com.trafficparrot.examples.grpc;

import com.trafficparrot.examples.grpc.ShippingOuterClass.*;
import com.trafficparrot.examples.tools.grpc.ShippingClient;
import io.grpc.StatusRuntimeException;
import org.junit.Test;

import static com.google.common.net.HostAndPort.fromParts;
import static com.trafficparrot.examples.tools.grpc.ShippingClient.createNonTlsShippingClient;
import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;

public class ManageStateIntegrationTest {
    private final ShippingClient shippingClient = createNonTlsShippingClient(fromParts("localhost", 5552));

    @Test
    public void simulatesServiceWithState() {
        CreateShippingDetailsRequest original = CreateShippingDetailsRequest.newBuilder()
                .setFirstName("first")
                .setMiddleName("middle")
                .setLastName("last")
                .setAddress(Address.newBuilder()
                        .setCountry("United Kingdom")
                        .setCity("London")
                        .setPostalCode("E18 1UX")
                        .setStreet1("1 gRPC road")
                        .setStreet2("Apartment 3")
                        .build())
                .build();
        String id = shippingClient.createShippingDetails(original);

        GetShippingDetailsResponse shippingDetails = shippingClient.getShippingDetails(id);
        assertThat(shippingDetails.getAddress()).isEqualTo(original.getAddress());
        assertThat(shippingDetails.getFirstName()).isEqualTo(original.getFirstName());
        assertThat(shippingDetails.getMiddleName()).isEqualTo(original.getMiddleName());
        assertThat(shippingDetails.getLastName()).isEqualTo(original.getLastName());

        ChangeNameRequest newName = ChangeNameRequest.newBuilder()
                .setFirstName("new first")
                .setMiddleName("new middle")
                .setLastName("new last")
                .setId(id)
                .build();
        shippingClient.changeName(newName);

        Address newAddress = Address.newBuilder()
                .setCountry("Brazil")
                .setCity("Rio de Janeiro")
                .setPostalCode("122231")
                .setStreet1("Av. das Américas, 19019")
                .setStreet2("Recreio dos Bandeirantes")
                .build();
        shippingClient.changeAddress(ChangeAddressRequest.newBuilder()
                .setId(id)
                .setAddress(newAddress)
                .build());

        GetShippingDetailsResponse newShippingDetails = shippingClient.getShippingDetails(id);
        assertThat(newShippingDetails.getAddress()).isEqualTo(newAddress);
        assertThat(newShippingDetails.getFirstName()).isEqualTo(newName.getFirstName());
        assertThat(newShippingDetails.getMiddleName()).isEqualTo(newName.getMiddleName());
        assertThat(newShippingDetails.getLastName()).isEqualTo(newName.getLastName());
        assertThat(newShippingDetails.getCreated()).isEqualTo(shippingDetails.getCreated());
        assertThat(newShippingDetails.getId()).isEqualTo(shippingDetails.getId());
    }

    @Test
    public void errorScenarioTakesPriority() {
        assertThatThrownBy(() -> shippingClient.createShippingDetails(CreateShippingDetailsRequest.getDefaultInstance()))
            .isInstanceOf(StatusRuntimeException.class)
            .hasMessage("INTERNAL: Last name is required");
    }
}
