package com.trafficparrot.examples.http;

import com.trafficparrot.examples.tools.http.HttpClient;
import org.assertj.core.api.Assertions;
import org.junit.Test;

import java.io.IOException;
import java.util.Base64;

/**
 * Example demonstrating Base64 XML XPath matching using existing WireMock Handlebars helpers.
 * 
 * This example shows how to:
 * - Extract base64-encoded XML from JSON requests using jsonPath helper
 * - Decode base64 content using base64 helper with decode=true
 * - Apply XPath queries to the decoded XML using xPath helper
 * - Use request matching based on XPath conditions in base64 XML content
 * 
 * No custom transformers are needed - everything is achieved through mapping configuration
 * using existing WireMock helpers: jsonPath, xPath, base64, matches
 */
public class Base64XmlHelperIntegrationTest {
    private final HttpClient httpClient = new HttpClient("http://localhost:18081");

    @Test(timeout = 30_000)
    public void extractXmlValueFromBase64JsonUsingResponseTemplating() throws IOException {
        httpClient.displayHttpMappingsContaining("base64xml");

        // Base64 encoded XML: <root><customer><id>12345</id><name>John Doe</name></customer></root>
        String xmlContent = "<root><customer><id>12345</id><name>John Doe</name></customer></root>";
        String base64XmlContent = Base64.getEncoder().encodeToString(xmlContent.getBytes());

        String requestJson = "{\"xmlData\":\"" + base64XmlContent + "\"}";
        String response = httpClient.post("/base64xml/extract", requestJson, "application/json");

        // Response template extracts customer data using: jsonPath -> base64 decode -> xPath
        Assertions.assertThat(response).contains("<customerId>12345</customerId>");
        Assertions.assertThat(response).contains("<customerName>John Doe</customerName>");
    }

    @Test(timeout = 30_000) 
    public void matchRequestBasedOnXPathInBase64XmlContent() throws IOException {
        httpClient.displayHttpMappingsContaining("base64xml");

        // Base64 encoded XML with VIP customer ID
        String xmlContent = "<root><customer><id>VIP001</id><name>Premium Customer</name></customer></root>";
        String base64XmlContent = Base64.getEncoder().encodeToString(xmlContent.getBytes());

        String requestJson = "{\"xmlData\":\"" + base64XmlContent + "\"}";
        String response = httpClient.post("/base64xml/service", requestJson, "application/json");

        // Request matcher script uses: jsonPath -> base64 decode -> xPath -> matches regex
        // Should match VIP pattern and return VIP service response
        Assertions.assertThat(response).contains("VIP Service: Welcome Premium Customer");
    }

    @Test(timeout = 30_000)
    public void matchStandardCustomer() throws IOException {
        httpClient.displayHttpMappingsContaining("base64xml");

        // Base64 encoded XML with standard customer ID
        String xmlContent = "<root><customer><id>STD002</id><name>Standard Customer</name></customer></root>";
        String base64XmlContent = Base64.getEncoder().encodeToString(xmlContent.getBytes());

        String requestJson = "{\"xmlData\":\"" + base64XmlContent + "\"}";
        String response = httpClient.post("/base64xml/service", requestJson, "application/json");

        // Should match standard customer pattern
        Assertions.assertThat(response).contains("Standard Service: Hello Standard Customer");
    }

    @Test(timeout = 30_000)
    public void encodeXmlResponseToBase64Json() throws IOException {
        httpClient.displayHttpMappingsContaining("base64xml");

        String response = httpClient.post("/base64xml/encode", "{\"request\":\"test\"}");

        // Response template uses base64 helper to encode XML response
        String expectedXml = "<response><data>processed</data></response>";
        String expectedBase64 = Base64.getEncoder().encodeToString(expectedXml.getBytes());
        Assertions.assertThat(response).contains("\"result\":\"" + expectedBase64 + "\"");
    }
}