package com.trafficparrot.examples.http;

import com.trafficparrot.examples.tools.http.HttpClient;
import org.assertj.core.api.Assertions;
import org.junit.Test;
import org.xmlunit.assertj.XmlAssert;

import java.io.IOException;


/**
 * Taken from the <a href="https://trafficparrot.com/tutorials/dynamic-responses.html">Dynamic Responses Tutorial</a>
 */
public class DynamicResponseIntegrationTest {
    private final HttpClient httpClient = new HttpClient("http://localhost:18081");

    @Test(timeout = 30_000)
    public void dynamicXmlResponse() throws IOException {
        httpClient.displayHttpMappingsContaining("stockRequest");

        // The #ifEven check returns stock level 5 if the checkId is odd
        String oddResponse = httpClient.post("/stockRequest", "<stockRequest>\n" +
                "    <checkId>1</checkId>\n" +
                "    <item>apple</item>\n" +
                "</stockRequest>");

        XmlAssert.assertThat(oddResponse).valueByXPath("/stockResponse/checkId").isEqualTo(1);
        XmlAssert.assertThat(oddResponse).valueByXPath("/stockResponse/date").matches("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}.*");
        XmlAssert.assertThat(oddResponse).valueByXPath("/stockResponse/item").isEqualTo("apple");
        XmlAssert.assertThat(oddResponse).valueByXPath("/stockResponse/available").isEqualTo(5);

        // The #ifEven check returns stock level 0 if the checkId is even
        String evenResponse = httpClient.post("/stockRequest", "<stockRequest>\n" +
                "    <checkId>2</checkId>\n" +
                "    <item>orange</item>\n" +
                "</stockRequest>");

        XmlAssert.assertThat(evenResponse).valueByXPath("/stockResponse/checkId").isEqualTo(2);
        XmlAssert.assertThat(evenResponse).valueByXPath("/stockResponse/date").matches("\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}.*");
        XmlAssert.assertThat(evenResponse).valueByXPath("/stockResponse/item").isEqualTo("orange");
        XmlAssert.assertThat(evenResponse).valueByXPath("/stockResponse/available").isEqualTo(0);
    }

    @Test(timeout = 30_000)
    public void xmlAttribute() throws IOException {
        httpClient.displayHttpMappingsContaining("requestXmlWithAttribute");

        XmlAssert.assertThat(httpClient.post("/requestXmlWithAttribute", "<request id=\"123\"/>")).isEqualTo("<response id=\"123\"/>");
        XmlAssert.assertThat(httpClient.post("/requestXmlWithAttribute", "<request id=\"456\"/>")).isEqualTo("<response id=\"456\"/>");
    }

    @Test(timeout = 30_000)
    public void xmlText() throws IOException {
        httpClient.displayHttpMappingsContaining("requestXmlWithText");

        String firstResponse = httpClient.post("/requestXmlWithText", "<request>\n" +
                "    <id>100</id>\n" +
                "    <name>example 1</name>\n" +
                "</request>");
        XmlAssert.assertThat(firstResponse).valueByXPath("/response/id").isEqualTo("100");
        XmlAssert.assertThat(firstResponse).valueByXPath("/response/name").isEqualTo("example 1");

        String secondResponse = httpClient.post("/requestXmlWithText", "<request>\n" +
                "    <id>333</id>\n" +
                "    <name>example 2</name>\n" +
                "</request>");
        XmlAssert.assertThat(secondResponse).valueByXPath("/response/id").isEqualTo("333");
        XmlAssert.assertThat(secondResponse).valueByXPath("/response/name").isEqualTo("example 2");
    }

    @Test(timeout = 30_000)
    public void jsonText() throws IOException {
        httpClient.displayHttpMappingsContaining("requestJsonWithText");

        String firstResponse = httpClient.post("/requestJsonWithText", "{\n" +
                "    \"type\": \"request\",\n" +
                "    \"id\": 100,\n" +
                "    \"name\": \"example 1\"\n" +
                "}");
        Assertions.assertThat(firstResponse).isEqualToIgnoringWhitespace("{\n" +
                "    \"type\": \"response\",\n" +
                "    \"id\": 100,\n" +
                "    \"name\": \"example 1\"\n" +
                "}");

        String secondResponse = httpClient.post("/requestJsonWithText", "{\n" +
                "    \"type\": \"request\",\n" +
                "    \"id\": 200,\n" +
                "    \"name\": \"example 2\"\n" +
                "}");
        Assertions.assertThat(secondResponse).isEqualToIgnoringWhitespace("{\n" +
                "    \"type\": \"response\",\n" +
                "    \"id\": 200,\n" +
                "    \"name\": \"example 2\"\n" +
                "}");
    }

    @Test(timeout = 30_000)
    public void xPathList() throws IOException {
        httpClient.displayHttpMappingsContaining("requestXPathList");

        String firstResponse = httpClient.post("/requestXPathList", "<request>\n" +
                "    <item id=\"1\"/>\n" +
                "    <item id=\"2\"/>\n" +
                "    <item id=\"3\"/>\n" +
                "</request>");
        Assertions.assertThat(firstResponse).isEqualToIgnoringWhitespace("<response>\n" +
                "    <item id=\"1\" status=\"OK\"/>\n" +
                "    <item id=\"2\" status=\"OK\"/>\n" +
                "    <item id=\"3\" status=\"OK\"/>\n" +
                "</response>");

        String secondResponse = httpClient.post("/requestXPathList", "<request>\n" +
                "    <item id=\"1\"/>\n" +
                "</request>");
        Assertions.assertThat(secondResponse).isEqualToIgnoringWhitespace("<response>\n" +
                "    <item id=\"1\" status=\"OK\"/>\n" +
                "</response>");
    }

    @Test(timeout = 30_000)
    public void jsonPathList() throws IOException {
        httpClient.displayHttpMappingsContaining("requestJsonPathList");

        String firstResponse = httpClient.post("/requestJsonPathList", "{\n" +
                "    \"type\": \"request\",\n" +
                "    \"items\": [\n" +
                "        { \"id\": 1 },\n" +
                "        { \"id\": 2 },\n" +
                "        { \"id\": 3 }\n" +
                "    ]\n" +
                "}");
        Assertions.assertThat(firstResponse).isEqualToIgnoringWhitespace("{\n" +
                "    \"type\": \"response\",\n" +
                "    \"items\": [\n" +
                "        { \"id\": 1, \"status\": \"OK\" },\n" +
                "        { \"id\": 2, \"status\": \"OK\" },\n" +
                "        { \"id\": 3, \"status\": \"OK\" }\n" +
                "    ]\n" +
                "}");

        String secondResponse = httpClient.post("/requestJsonPathList", "{\n" +
                "    \"type\": \"request\",\n" +
                "    \"items\": [\n" +
                "        { \"id\": 1 }\n" +
                "    ]\n" +
                "}");
        Assertions.assertThat(secondResponse).isEqualToIgnoringWhitespace("{\n" +
                "    \"type\": \"response\",\n" +
                "    \"items\": [\n" +
                "        { \"id\": 1, \"status\": \"OK\" }\n" +
                "    ]\n" +
                "}");
    }
}
