package com.trafficparrot.examples.tools.grpc;

import com.google.common.net.HostAndPort;
import com.trafficparrot.examples.grpc.ShippingGrpc;
import com.trafficparrot.examples.grpc.ShippingOuterClass;
import com.trafficparrot.examples.grpc.ShippingOuterClass.ChangeAddressRequest;
import com.trafficparrot.examples.grpc.ShippingOuterClass.ChangeNameRequest;
import com.trafficparrot.examples.grpc.ShippingOuterClass.CreateShippingDetailsRequest;
import com.trafficparrot.examples.grpc.ShippingOuterClass.GetShippingDetailsResponse;
import io.grpc.ManagedChannel;
import io.grpc.netty.NettyChannelBuilder;
import io.netty.handler.ssl.SslContext;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import static io.grpc.netty.NegotiationType.PLAINTEXT;
import static io.grpc.netty.NegotiationType.TLS;

public class ShippingClient {

    private final ManagedChannel channel;
    private final ShippingGrpc.ShippingBlockingStub shippingService;

    private ShippingClient(ManagedChannel channel, ShippingGrpc.ShippingBlockingStub shippingService) {
        this.channel = channel;
        this.shippingService = shippingService;
    }

    public static ShippingClient createTlsShippingClient(HostAndPort hostAndPort, SslContext sslContext, String overrideAuthority) {
        return createOrderClient(hostAndPort, Optional.of(sslContext), overrideAuthority);
    }

    public static ShippingClient createNonTlsShippingClient(HostAndPort hostAndPort) {
        return createOrderClient(hostAndPort, Optional.empty(), "localhost");
    }

    private static ShippingClient createOrderClient(HostAndPort hostAndPort, Optional<SslContext> sslContext, String overrideAuthority) {
        NettyChannelBuilder builder = NettyChannelBuilder.forAddress(hostAndPort.getHost(), hostAndPort.getPort())
                .overrideAuthority(overrideAuthority)
                .negotiationType(sslContext.map(x -> TLS).orElse(PLAINTEXT));
        sslContext.ifPresent(builder::sslContext);
        ManagedChannel channel = builder.build();

        ShippingGrpc.ShippingBlockingStub shippingService = ShippingGrpc.newBlockingStub(channel);
        return new ShippingClient(channel, shippingService);
    }

    public void shutdown() throws InterruptedException {
        channel.shutdown().awaitTermination(5, TimeUnit.SECONDS);
    }

    public String createShippingDetails(CreateShippingDetailsRequest request) {
        return shippingService.createShippingDetails(request).getId();
    }

    public void changeAddress(ChangeAddressRequest request) {
        shippingService.changeAddress(request);
    }

    public void changeName(ChangeNameRequest request) {
        shippingService.changeName(request);
    }

    public GetShippingDetailsResponse getShippingDetails(String id) {
        return shippingService.getShippingDetails(ShippingOuterClass.GetCustomerRequest.newBuilder().setId(id).build());
    }
}

