package com.trafficparrot.sdk.example.http;

import com.github.tomakehurst.wiremock.common.FileSource;
import com.github.tomakehurst.wiremock.common.Gzip;
import com.github.tomakehurst.wiremock.extension.Parameters;
import com.github.tomakehurst.wiremock.http.*;
import com.trafficparrot.sdk.http.HttpResponseTransformer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import static com.github.tomakehurst.wiremock.http.HttpHeader.httpHeader;
import static com.github.tomakehurst.wiremock.http.Response.Builder.like;
import static com.trafficparrot.sdk.example.http.MoveDatesInXmlRequestTransformer.FOUR_YEARS_IN_MILLIS;
import static com.trafficparrot.sdk.http.TransformedRequest.contentEncoding;

/**
 *  Use together with {@link MoveDatesInXmlRequestTransformer}
 *
 *  It accepts response body:
 *
 *  <ns:transactionsResult xmlns:ns="http://example.trafficparrot.com"><transaction><day>2020-01-03</day><user></user></transaction></ns:transactionsResult>
 */
public class MoveDatesInXmlResponseTransformer extends HttpResponseTransformer {
    private static final Logger LOGGER = LoggerFactory.getLogger(MoveDatesInXmlResponseTransformer.class);

    public static final HttpHeader TRANSFORMER_ADDED_RESPONSE_HEADER = httpHeader("transformed-and-proxied-by", MoveDatesInXmlResponseTransformer.class.getName());

    @Override
    protected Response doTransform(Request request, Response originalResponse, FileSource fileSource, Parameters parameters) {
        LOGGER.info("Transformer received response " + originalResponse);
        LOGGER.info("Original response body " + originalResponse.getBodyAsString());
        byte[] transformedBody = moveDatesInResponseBody(originalResponse.getBody(), originalResponse.getHeaders());
        Response transformedResponse = like(originalResponse)
                .body(transformedBody)
                .headers(originalResponse.getHeaders().plus(TRANSFORMER_ADDED_RESPONSE_HEADER))
                .build();
        LOGGER.info("Transformed response body " + transformedResponse.getBodyAsString());
        LOGGER.info("Transformer returning response " + transformedResponse);
        return transformedResponse;
    }

    private byte[] moveDatesInResponseBody(byte[] requestMessageBody, HttpHeaders originalResponseHeaders) {
        HttpHeader contentEncoding = originalResponseHeaders.getHeader("content-encoding");
        if (contentEncoding.isPresent() && contentEncoding.values().contains("gzip")) {
            byte[] unpackedResponseBody = Gzip.unGzip(requestMessageBody);
            Charset responseCharset = contentEncoding(originalResponseHeaders.getContentTypeHeader());
            byte[] transformedResponse = moveDatesInResponseBody(new String(unpackedResponseBody, responseCharset)).getBytes(responseCharset);
            return Gzip.gzip(transformedResponse);
        } else {
            return moveDatesInResponseBody(requestMessageBody, originalResponseHeaders.getContentTypeHeader());
        }
    }

    private byte[] moveDatesInResponseBody(byte[] requestMessageBody, ContentTypeHeader contentTypeHeader) {
        Charset responseCharset = contentEncoding(contentTypeHeader);
        return moveDatesInResponseBody(new String(requestMessageBody, responseCharset)).getBytes(responseCharset);
    }

    private String moveDatesInResponseBody(String requestMessageBody) {
        try {
            // parse the XML request body
            DocumentBuilderFactory abstractFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder factory = abstractFactory.newDocumentBuilder();
            Document doc = factory.parse(new ByteArrayInputStream(requestMessageBody.getBytes()));
            String requestDateString = XPathFactory.newInstance().newXPath().compile("/transactionsResult/transaction/day/text()").evaluate(doc);
            SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
            Date requestDate = format.parse(requestDateString);

            // calculate the new date (in the present)
            Date movedDate = new Date(requestDate.getTime() + FOUR_YEARS_IN_MILLIS);
            String movedDateString = format.format(movedDate);

            // set new date
            return requestMessageBody.replace(requestDateString, movedDateString);
        } catch (ParserConfigurationException | SAXException | IOException | XPathExpressionException | ParseException e) {
            throw new IllegalArgumentException(e);
        }
    }

    @Override
    public String getName() {
        return getClass().getSimpleName();
    }
}
