package com.trafficparrot.sdk.example.nativeibmmq;

import com.github.tomakehurst.wiremock.extension.Parameters;
import com.ibm.mq.MQMessage;
import com.trafficparrot.sdk.ibmmq.IbmMqResponse;
import com.trafficparrot.sdk.ibmmq.IbmMqResponseTransformer;
import com.trafficparrot.sdk.jms.Destination;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Optional;
import java.util.Properties;

import static com.trafficparrot.sdk.PropertiesHelper.readPropertiesFile;
import static com.trafficparrot.sdk.ibmmq.DestinationLookup.USE_MAPPING_RESPONSE_DESTINATION;
import static com.trafficparrot.sdk.ibmmq.SdkIbmMqUtils.mqMessageToString;

/**
 * This is a sample passthrough proxy message transformer.
 *
 * Without Traffic Parrot, assume there are two queues REAL.REQUEST.QUEUE and REAL.RESPONSE.QUEUE that the SUT uses to
 * communicate with another system DEP such that messages flow:
 * SUT -> REAL.REQUEST.QUEUE -> DEP -> REAL.RESPONSE.QUEUE -> SUT
 *
 * We insert proxy queues such that messages flow this way when the proxy mappings are being used:
 * SUT -> PROXY.REQUEST.QUEUE -> TP -> REAL.REQUEST.QUEUE -> DEP
 * DEP -> REAL.RESPONSE.QUEUE -> TP -> PROXY.RESPONSE.QUEUE -> SUT
 *
 * The properties file ibm-mq-passthrough-proxy.properties contains:
 * REAL.REQUEST.QUEUE.replyToQueueName=REAL.RESPONSE.QUEUE
 * PROXY.RESPONSE.QUEUE.replyToQueueName=PROXY.REQUEST.QUEUE
 *
 * Which ensures that messages can flow between SUT AND DEP when using the proxy mappings.
 *
 * When mock mappings are being used, messages instead flow this way:
 * SUT -> PROXY.REQUEST.QUEUE -> TP -> PROXY.RESPONSE.QUEUE -> SUT
 *
 * This transformer can be configured with request mapping:
 *
 *  {
 *   "mappingId" : "proxy-request-messages-mapping-1001",
 *   "request" : {
 *     "destination" : {
 *       "name" : "PROXY.REQUEST.QUEUE",
 *       "type" : "QUEUE"
 *     },
 *     "bodyMatcher" : {
 *       "anything" : "anything"
 *     }
 *   },
 *   "response" : {
 *     "destination" : {
 *       "name" : "REAL.REQUEST.QUEUE",
 *       "type" : "QUEUE"
 *     },
 *     "ibmMqResponseTransformerClassName" : "com.trafficparrot.sdk.example.nativeibmmq.PassthroughMessage"
 *   },
 *   "receiveThreads" : 1,
 *   "sendThreads" : 1
 *  }
 *
 * And corresponding response mapping:
 *  {
 *   "mappingId" : "proxy-response-messages-mapping-1001",
 *   "request" : {
 *     "destination" : {
 *       "name" : "REAL.RESPONSE.QUEUE",
 *       "type" : "QUEUE"
 *     },
 *     "bodyMatcher" : {
 *       "anything" : "anything"
 *     }
 *   },
 *   "response" : {
 *     "destination" : {
 *       "name" : "PROXY.RESPONSE.QUEUE",
 *       "type" : "QUEUE"
 *     },
 *     "ibmMqResponseTransformerClassName" : "com.trafficparrot.sdk.example.nativeibmmq.PassthroughMessage"
 *   },
 *   "receiveThreads" : 1,
 *   "sendThreads" : 1
 *  }
 */
public class PassthroughMessage extends IbmMqResponseTransformer {
    private static final Logger LOGGER = LoggerFactory.getLogger(PassthroughMessage.class);

    public static final String PASSTHROUGH_PROXY_PROPERTIES = "ibm-mq-passthrough-proxy.properties";

    // You can put this file in the main Traffic Parrot directory where all the other properties files are
    private final Properties properties = readPropertiesFile(PASSTHROUGH_PROXY_PROPERTIES);

    @Override
    @SuppressWarnings("UnnecessaryLocalVariable")
    protected IbmMqResponse doTransform(Destination requestDestination, MQMessage requestMessage, IbmMqResponse mappingResponse, Parameters parameters) throws Exception {
        MQMessage passthroughMessage = requestMessage;
        setReplyToQ(passthroughMessage, mappingResponse.destination);
        resetMqPropertiesThatAreSetByMq(passthroughMessage);
        return new IbmMqResponse(mappingResponse.destination, passthroughMessage, mappingResponse.fixedDelayMilliseconds, Optional.of(USE_MAPPING_RESPONSE_DESTINATION));
    }

    private void setReplyToQ(MQMessage passThroughMessage, Destination passthroughDestination) {
        passThroughMessage.replyToQueueName = getReplyToQueue(passThroughMessage, passthroughDestination);
    }

    private String getReplyToQueue(MQMessage passthroughMessage, Destination passthroughDestination) {
        String messageAsString = mqMessageToString(passthroughMessage);
        LOGGER.info("Transformer message converted to string: " + messageAsString);
        String key = passthroughDestination.name.toUpperCase().trim() + ".replyToQueueName";
        String passthroughQueue = properties.getProperty(key);
        if (passthroughQueue == null) {
            throw new IllegalStateException("Unable to determine replyToQueueName for: " + key);
        }
        LOGGER.info("Transformer using replyToQueueName: " + passthroughQueue);
        return passthroughQueue;
    }

    private void resetMqPropertiesThatAreSetByMq(MQMessage passthroughMessage) {
        passthroughMessage.putDateTime = null;
        passthroughMessage.putApplicationName = "Traffic Parrot transformer proxy " + getClass().getSimpleName();
    }
}
