package com.trafficparrot.sdk.example.nativeibmmq;

import com.github.tomakehurst.wiremock.extension.Parameters;
import com.ibm.mq.MQMessage;
import com.trafficparrot.sdk.ibmmq.IbmMqResponse;
import com.trafficparrot.sdk.ibmmq.IbmMqResponseTransformer;
import com.trafficparrot.sdk.jms.Destination;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Optional;
import java.util.Properties;

import static com.trafficparrot.sdk.PropertiesHelper.readPropertiesFile;
import static com.trafficparrot.sdk.example.nativeibmmq.TransformAndProxyRequestMessage.TRANSFORM_AND_PROXY_PROPERTIES_FILENAME;
import static com.trafficparrot.sdk.ibmmq.DestinationLookup.USE_MAPPING_RESPONSE_DESTINATION;
import static com.trafficparrot.sdk.ibmmq.SdkIbmMqUtils.readMessageBodySkipHeaders;
import static com.trafficparrot.sdk.ibmmq.SdkIbmMqUtils.setMessageBody;

/**
 * This is a sample proxy response message transformer. It reads its configuration from a properties file.
 *
 * This transformer can be configured with mapping:
 *
 * {
 *   "mappingId" : "proxy-response-messages-mapping-1002",
 *   "request" : {
 *     "destination" : {
 *       "name" : "REAL.RESPONSE.QUEUE",
 *       "type" : "QUEUE"
 *     },
 *     "bodyMatcher" : {
 *       "anything" : "anything"
 *     }
 *   },
 *   "response" : {
 *     "destination" : {
 *       "name" : "PROXY.RESPONSE.QUEUE",
 *       "type" : "QUEUE"
 *     },
 *     "ibmMqResponseTransformerClassName" : "com.trafficparrot.sdk.example.nativeibmmq.TransformAndProxyResponseMessage"
 *   },
 *   "receiveThreads" : 1,
 *   "sendThreads" : 1
 * }
 *
 *
 *
 * It accepts message body:
 *
 * <ns:transactionsResult xmlns:ns="http://example.trafficparrot.com"><transaction><day>2020-01-03</day><user></user></transaction></ns:transactionsResult>
 */
public class TransformAndProxyResponseMessage extends IbmMqResponseTransformer {
    private static final long FOUR_YEARS_IN_MILLIS = 4L * 365 * 24 * 60 * 60 * 1000;

    public static final String PROXY_REQUEST_QUEUE_KEY = "proxy.request.queue";

    // You can put this file in the main Traffic Parrot directory where all the other properties files are
    private final Properties properties = readPropertiesFile(TRANSFORM_AND_PROXY_PROPERTIES_FILENAME);

    @Override
    @SuppressWarnings("UnnecessaryLocalVariable")
    protected IbmMqResponse doTransform(Destination requestDestination, MQMessage requestMessage, IbmMqResponse mappingResponse, Parameters parameters) throws Exception {
        MQMessage proxyResponseMessage = requestMessage;
        setReplyToQ(proxyResponseMessage);
        resetMqPropertiesThatAreSetByMq(proxyResponseMessage);
        moveDatesInBody(requestMessage, proxyResponseMessage);
        return new IbmMqResponse(mappingResponse.destination, proxyResponseMessage, mappingResponse.fixedDelayMilliseconds, Optional.of(USE_MAPPING_RESPONSE_DESTINATION));
    }

    private void setReplyToQ(MQMessage proxyResponseMessage) {
        proxyResponseMessage.replyToQueueName = properties.getProperty(PROXY_REQUEST_QUEUE_KEY);
    }

    private void moveDatesInBody(MQMessage requestMessage, MQMessage proxyResponseMessage) throws IOException, ParserConfigurationException, XPathExpressionException, SAXException, ParseException {
        // parse the XML request body
        String originalMessageBody = readMessageBodySkipHeaders(requestMessage);
        DocumentBuilderFactory abstractFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder factory = abstractFactory.newDocumentBuilder();
        Document doc = factory.parse(new ByteArrayInputStream(originalMessageBody.getBytes()));
        String requestDateString = XPathFactory.newInstance().newXPath().compile("/transactionsResult/transaction/day/text()").evaluate(doc);
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Date requestDate = format.parse(requestDateString);

        // calculate the new date (in the present)
        Date movedDate = new Date(requestDate.getTime() + FOUR_YEARS_IN_MILLIS);
        String movedDateString = format.format(movedDate);

        // set new date in the response
        String newMessageBody = originalMessageBody.replace(requestDateString, movedDateString);
        setMessageBody(proxyResponseMessage, newMessageBody);
    }

    private void resetMqPropertiesThatAreSetByMq(MQMessage proxyResponseMessage) {
        proxyResponseMessage.putDateTime = null;
        proxyResponseMessage.putApplicationName = "Traffic Parrot transformer proxy " + getClass().getSimpleName();
    }
}
